# (c) Copyright 2012, 2016, 2020-2021. CodeWeavers, Inc.

"""Provides efficient access to the cxdiag diagnostics."""

import os.path
import time
import re

import cxconfig
import cxlog
import cxproduct
import cxutils
import cxwhich
import distversion

# for localization
from cxutils import cxgettext as _

import bottlequery

CHECK_NOBIT = 1
CHECK_32BIT = 2
CHECK_64BIT = 4
CHECK_PKGDEB = 16
CHECK_PKGRPM = 32
CHECK_OPENSSL = 64
CHECK_PERLFILECOPY = 128
CHECK_GZIP = 256
CHECK_PKEXEC = 512


#
# The cxdiag cache
#

_CACHE = {}

def _get_cached_cxdiag(libpath, flags):
    cxdiag = _CACHE.get((libpath, flags), None)
    if cxdiag and time.time()-cxdiag.timestamp > 60:
        # Remove old results in case the user added / removed packages in the
        # meantime
        cxlog.log("cxdiag: pruning obsolete results")
        del _CACHE[(libpath, flags)]
        cxdiag = None
    return cxdiag


#
# CXDiag class
#

_LOCALIZE_LIB_RE = re.compile('^(?P<type>Broken|Missing) (?P<bitness>32|64)-bit (?P<library>.+) library$')
_LOCALIZE_GST_RE = re.compile('^The (?P<module>.+) (?P<bitness>32|64)-bit GStreamer plugins appear to be missing (?P<plugin>.+)$')

def localize_title(title):
    match = _LOCALIZE_LIB_RE.match(title)
    if match:
        if match.group('type') == 'Broken':
            return _('Broken %(bitness)s-bit %(library)s library') % {'bitness': match.group('bitness'), 'library': match.group('library')}
        return _('Missing %(bitness)s-bit %(library)s library') % {'bitness': match.group('bitness'), 'library': match.group('library')}
    match = _LOCALIZE_GST_RE.match(title)
    if match:
        return _("The %(module)s %(bitness)s-bit GStreamer plugins appear to be missing %(plugin)s") % {'module': match.group('module'), 'bitness': match.group('bitness'), 'plugin': match.group('plugin')}
    return cxutils.cxgettext(title)

class CXDiag:
    """Stores the cxdiag warnings and properties for the given library path
    setting.

    Once a CXDiag object is created it is never updated. So users should not
    keep references to them but instead systematically call cxdiag.get().

    The warnings are stored in the warnings dictionary which maps issue ids
    to their already localized titles.
    The properties are stored in the properties dictionary.
    """

    def _run_cxdiag(self, env, bitness, bitsuffix):
        cxdiag = "cxdiag" + bitsuffix
        cxdiagpath = os.path.join(cxutils.CX_ROOT, "bin", cxdiag)
        retcode, out, _err = cxutils.run((cxdiagpath, ), env=env,
                                         stdout=cxutils.GRAB)
        if retcode == 0:
            config = cxconfig.Raw()
            config.read_string(out)
            for name, section in config.items():
                name = name.lower()
                if name == 'properties':
                    for prop, value in section.items():
                        self.properties[prop.lower()] = value
                else:
                    title = localize_title(section['Title'])
                    self.warnings[name] = title
                    if 'Lib' in section:
                        self.libs[name] = section['Lib']
        elif not os.access(cxdiagpath, os.R_OK | os.X_OK):
            # Consider that cxdiag has been disabled
            cxlog.log("%s is missing / not executable" % cxdiag)
        elif retcode == -1:
            cxlog.warn("could not run %s" % cxdiag)
            self.warnings['missinglibc'] = _('Missing %(bitness)s-bit %(library)s library') % {'bitness': bitness, 'library': 'C'}
        else:
            cxlog.warn("%s failed (%d)" % (cxdiag, retcode))
            self.warnings['broken' + cxdiag] = _('%s failed' % cxdiag)

    def _get_cxdiag(self, libpath, flags, bitflag):
        if flags != bitflag:
            # Merge the standalone 32- or 64-bit cxdiag results
            cxdiag = _get_cached_cxdiag(libpath, bitflag)
            if cxdiag is None:
                cxdiag = CXDiag(libpath, bitflag)
            self.libs.update(cxdiag.libs)
            self.warnings.update(cxdiag.warnings)
            self.properties.update(cxdiag.properties)
            self.timestamp = min(self.timestamp, cxdiag.timestamp) # Set the timestamp to the oldest data
        else:
            env = os.environ.copy()
            if libpath:
                # Re-run cxdiag with the new library path.
                varname = "LD_LIBRARY_PATH"
                if varname not in env:
                    env[varname] = libpath
                else:
                    env[varname] = ':'.join((libpath, env[varname]))
                cxlog.log("running cxdiag* with %s=%s" % (varname, libpath))

            if bitflag == CHECK_32BIT:
                self._run_cxdiag(env, "32", "")
            else:
                self._run_cxdiag(env, "64", "64")

    @classmethod
    def _has_tools(cls, tools):
        for tool in tools:
            if cxwhich.which(os.environ["PATH"], tool) is None:
                return False
        return True

    def _check_pkgdeb(self):
        if not CXDiag._has_tools(('dh_builddeb', 'dpkg-buildpackage', 'fakeroot')):
            self.warnings['packagingdeb'] = _('Missing Debian packaging tools')

    def _check_pkgrpm(self):
        if not CXDiag._has_tools(('rpmbuild',)):
            self.warnings['packagingrpm'] = _('Missing RPM packaging tools')

    def _check_openssl(self):
        if not CXDiag._has_tools(('openssl',)):
            self.warnings['missingopenssl'] = _('Missing openssl tool')

    def _check_perlfilecopy(self):
        if os.system("perl -MFile::Copy -e 1 2>/dev/null") != 0:
            self.warnings['missingperlfilecopy'] = _("Missing File::Copy Perl module")

    def _check_pkexec(self):
        if not CXDiag._has_tools(('pkexec',)):
            self.warnings['missingpkexec'] = _('Missing pkexec tool')

    def _check_gzip(self):
        if not CXDiag._has_tools(('gzip',)):
            self.warnings['missinggzip'] = _('Missing gzip tool')

    def __init__(self, libpath, flags):
        # Set up the library path so cxdiag takes into account the builtin
        # libraries. Note that it is configurable per bottle.
        self.libpath = libpath
        self.libs = {}
        self.warnings = {}
        self.properties = {}
        self.timestamp = time.time()

        if not distversion.IS_MACOSX:
            if flags & CHECK_32BIT:
                self._get_cxdiag(libpath, flags, CHECK_32BIT)
            if flags & CHECK_64BIT:
                self._get_cxdiag(libpath, flags, CHECK_64BIT)
            if flags & CHECK_PKGDEB:
                self._check_pkgdeb()
            if flags & CHECK_PKGRPM:
                self._check_pkgrpm()
            if flags & CHECK_OPENSSL:
                self._check_openssl()
            if flags & CHECK_GZIP:
                self._check_gzip()
            if flags & CHECK_PERLFILECOPY:
                self._check_perlfilecopy()
            if flags & CHECK_PKEXEC:
                self._check_pkexec()

        # Update the cache
        # pylint: disable=W0603
        global _CACHE
        _CACHE[(libpath, flags)] = self


#
# Public API
#

def get(bottlename=None, flags=0):
    """Return a CXDiag object with the results for the specified bottle.

    The results are cached so we don't run cxdiag a dozen times if we have
    half a dozen bottles. However they are also expired after a while so we
    detect configuration changes such as the addition or removal of packages.
    """
    # The cxdiag result depends on the configuration settings, specifically
    # on the LibPath setting.
    if bottlename:
        config = bottlequery.get_config(bottlename)
    else:
        config = cxproduct.get_config()
    libpath = config['Wine'].get('LibPath', None)
    if libpath is not None:
        libpath = bottlequery.expand_unix_string(bottlename, libpath)
    else:
        libpath = os.path.join(cxutils.CX_ROOT, "lib")
    if flags & (CHECK_NOBIT | CHECK_32BIT | CHECK_64BIT):
        # The caller already specified which bitnesses to check
        pass
    elif bottlename:
        flags |= CHECK_32BIT # we don't have 64-bit-only bottles
        if config["Bottle"].get("WineArch", "") == "win64":
            flags |= CHECK_64BIT
    else:
        if cxproduct.is_32bit_install():
            flags |= CHECK_32BIT
        if cxproduct.is_64bit_install():
            flags |= CHECK_64BIT

    cxdiag = _get_cached_cxdiag(libpath, flags)
    if cxdiag is None:
        cxdiag = CXDiag(libpath, flags)
    return cxdiag

def clear():
    """Clears the cxdiag cache.

    This forces redetecting issues. This should be called whenever packages
    have been installed to fix issues.
    """
    cxlog.log("cxdiag: clearing obsolete results")
    # pylint: disable=W0603
    global _CACHE
    _CACHE = {}
