# (c) Copyright 2014-2021. CodeWeavers, Inc.

import collections
import os.path
import re
import subprocess

import cxlog
import cxproduct
import cxutils
import distversion

# for localization
from cxutils import cxgettext as _

FIXES_URL = "http://ftp.codeweavers.com/pub/crossover/cxfixes/cxfixes.xml.gz"
WIKI_URL = "https://www.codeweavers.com/support/wiki/Diag/"


#####
#
# Methods for sequentially running a Gtk UI from the main thread
#
#####

_LOCK = None

def run_from_main_thread_callback():
    from gi.repository import Gtk

    if _LOCK:
        _LOCK.release()
    else:
        Gtk.main_quit()

def run_from_main_thread(fun):
    # pylint: disable=W0603
    global _LOCK

    import checkgtk
    import threading
    from gi.repository import Gtk
    from gi.repository import Gdk

    gtk_level = Gtk.main_level()
    if gtk_level != 0:
        _LOCK = threading.Lock()
        _LOCK.acquire() # pylint: disable=R1732

    Gdk.threads_add_idle(0, fun, None)

    if gtk_level == 0:
        if not checkgtk.init_check():
            raise Exception('No display found')

        Gtk.main()

    if gtk_level != 0:
        _LOCK.acquire()
        _LOCK = None


#####
#
# Methods for displaying error messages
#
#####

def display_error_internal(title, text):
    text = markup_error_message(text)
    try:
        from gi.repository import Gtk
        import cxguitools

        def done(_response):
            run_from_main_thread_callback()

        def create_dialog(_data):
            cxguitools.CXMessageDlg(
                primary=title,
                markup=text,
                button_array=((Gtk.STOCK_OK, 0),),
                response_function=done,
                message_type=Gtk.MessageType.ERROR)

        run_from_main_thread(create_dialog)

        return 0
    except:
        return -1

def display_error_zenity(title, text):
    title = cxutils.string_to_str(title)
    text = cxutils.string_to_str(text)
    return cxutils.run(['zenity', '--error', '--no-markup', '--title', title,
                        '--text', text])

def display_error_kdialog(title, text):
    title = cxutils.string_to_str(title)
    text = cxutils.string_to_str(text)
    return cxutils.run(['kdialog', '--title', title, '--error', text])

def display_error_xmessage(title, text):
    if not isinstance(title, str):
        try:
            title = title.encode('iso-8859-1')
        except UnicodeEncodeError:
            # Hope xmessage will display it right anyway
            title = title.encode('utf8')
    if not isinstance(text, str):
        try:
            text = text.encode('iso-8859-1')
        except UnicodeEncodeError:
            # Hope xmessage will display it right anyway
            text = text.encode('utf8')
    return cxutils.run(['xmessage', title + '\n' + text])

def display_error(gui, title, text):
    if not gui:
        # pylint: disable=C0325
        print(text)
        return

    # kdialog and xmessage return 1 either when an error occurs or when the user
    # closes the window instead of clicking on a button. So we cannot know if
    # the user really got notified through the GUI. So always print the error
    # on the console too. Do it first too in case the GUI garbles the message
    # (xmessage I'm looking at you).
    import sys
    sys.stderr.write(title + "\n" + text)

    # retcode is negative if the tool is not present, except for zenity which
    # also returns -5 if it cannot open the display. Either way it means the
    # user did not get the warning so try the next one when that happens.
    retcode = display_error_internal(title, text)
    if retcode < 0:
        retcode, _out, _err = display_error_zenity(title, text)
    if retcode < 0:
        retcode, _out, _err = display_error_kdialog(title, text)
    if retcode < 0:
        # xmessage should only be used as a last resort
        display_error_xmessage(title, text)


#####
#
# Methods for asking yes / no questions
#
#####

def refused_internal(title, text, prompt):
    ret = [1]
    try:
        from gi.repository import Gtk
        import cxguitools

        def done(response, userdata):
            userdata[0] = response
            run_from_main_thread_callback()

        def create_dialog(_data):
            cxguitools.CXMessageDlg(
                primary=title,
                secondary=text + '\n\n' + prompt,
                button_array=((Gtk.STOCK_NO, 1), (Gtk.STOCK_YES, 0)),
                response_function=done,
                user_data=ret,
                message_type=Gtk.MessageType.QUESTION)

        run_from_main_thread(create_dialog)

        return ret[0]
    except:
        return -1

def refused_zenity(title, text, prompt):
    title = cxutils.string_to_str(title)
    text = cxutils.string_to_str(text + '\n\n' + prompt)
    return cxutils.run(['zenity', '--question', '--no-markup', '--title', title,
                        '--text', text])

def refused_kdialog(title, text, prompt):
    title = cxutils.string_to_str(title)
    text = cxutils.string_to_str(text + '\n\n' + prompt)
    return cxutils.run(['kdialog', '--title', title, '--yesno', text])

def refused_xmessage(title, text, prompt):
    if not isinstance(title, str):
        try:
            title = title.encode('iso-8859-1')
        except UnicodeEncodeError:
            # Hope xmessage will display it right anyway
            title = title.encode('utf8')
    text += prompt
    if not isinstance(text, str):
        try:
            text = text.encode('iso-8859-1')
        except UnicodeEncodeError:
            # Hope xmessage will display it right anyway
            text = text.encode('utf8')
    if not isinstance(prompt, str):
        try:
            prompt = prompt.encode('iso-8859-1')
        except UnicodeEncodeError:
            # Hope xmessage will display it right anyway
            prompt = prompt.encode('utf8')
    return cxutils.run(['xmessage', '-buttons', 'No:1,Yes:0', '-default', 'Yes',
                        title + '\n' + text + '\n\n' + prompt])

def refused(gui, title, text, prompt):
    retcode = -1
    if not gui:
        # pylint: disable=C0325
        print(title)
        print(text)
        try:
            r = input(prompt + " [Y/n] ")
            retcode = 0 if r in ('', 'y', 'Y') else 1
        except EOFError:
            retcode = -2

    # The GUI tools return 1 either when an error occurs or when the user
    # closes the window instead of clicking on a button. So we cannot know if
    # the user really got notified through the GUI. So always print the error
    # on the console too. Do it first too in case the GUI garbles the message
    # (xmessage I'm looking at you).
    if retcode == -1:
        import sys
        sys.stderr.write(title + "\n" + text+'\n')

    # retcode is negative if the tool is not present, except for zenity which
    # also returns -5 if it cannot open the display. Either way it means the
    # user did not get the warning so try the next one when that happens.
    if retcode == -1:
        retcode = refused_internal(title, text, prompt)
    if retcode < 0:
        retcode, _out, _err = refused_zenity(title, text, prompt)
    if retcode < 0:
        retcode, _out, _err = refused_kdialog(title, text, prompt)
    if retcode < 0:
        retcode, _out, _err = refused_xmessage(title, text, prompt)
    return retcode


#####
#
# Loading the distributions and fixes information
#
#####

CXFIXES = None
UPDATE = True

def _load_fixes():
    # pylint: disable=W0603
    global CXFIXES

    if CXFIXES is not None:
        return

    # Try to grab the latest information on distributions and fixes. Use a
    # short timeout to not risk delaying the application startup too much.
    latestfixes = os.path.join(cxproduct.get_user_dir(), "cxfixes.xml")
    if UPDATE:
        try:
            import fileupdate
            fileupdate.update(latestfixes, FIXES_URL, True, 5)
        except:
            pass

    import cxfixesparser
    latest, errors = cxfixesparser.read_file(latestfixes)
    for error in errors:
        cxlog.warn(error)

    # Load the builtin information
    builtin, errors = cxfixesparser.read_file(os.path.join(cxutils.CX_ROOT, "share/crossover/data/cxfixes.xml"))
    for error in errors:
        cxlog.warn(error)

    # Use the most recent data
    if builtin is None and latest is None:
        cxlog.warn("Could not load any fixes information")
        import cxfixesobjs
        CXFIXES = cxfixesobjs.CXFixes()
    elif builtin is None or (latest is not None and builtin.release < latest.release):
        cxlog.log("using the downloaded fixes information (release %s)" % latest.release)
        latest.prepare()
        CXFIXES = latest
    else:
        cxlog.log("using the builtin fixes information (release %s)" % builtin.release)
        builtin.prepare()
        CXFIXES = builtin


#####
#
# Platform detection
#
#####

_BITNESS = None

def detect_bitness():
    # pylint: disable=W0603
    global _BITNESS
    if _BITNESS is None:
        # We don't detect bitness through uname to correctly deal with chroots.
        _BITNESS = '64' # default to 64-bit
        with open('/bin/ls', 'rb') as f:
            if f.read(5) == b'\x7fELF\x01':
                _BITNESS = '32'
    return _BITNESS

_DISTROID = None

def detect_distribution():
    # pylint: disable=W0603
    global _DISTROID
    if _DISTROID is None:
        _load_fixes()
        import globtree
        glob_tree = globtree.FileContentGlobTree()

        for (distroid, distro) in CXFIXES.distributions.items():
            for distglob in distro.globs:
                try:
                    glob_tree.add_content_glob(distglob.file_glob, distglob.patterns, distroid)
                except re.error:
                    cxlog.warn("the %s fileglob of %s contains an invalid regular expression" % (distglob.file_glob, distroid))


        best_priority = -1
        for _filename, distroid in glob_tree.matches(b'/'):
            if _DISTROID is None or \
                    best_priority < CXFIXES.distributions[distroid].priority:
                _DISTROID = distroid
                best_priority = CXFIXES.distributions[distroid].priority
                cxlog.log("found a match for distro %s (%s)" % (_DISTROID, best_priority))
    if _DISTROID is None:
        cxlog.err("Could not identify the distribution")
    return _DISTROID

_CXID = None

def detect_cxproduct():
    # pylint: disable=W0603
    global _CXID
    if _CXID is None:
        _load_fixes()
        cx_product_id = distversion.BUILTIN_PRODUCT_ID
        cx_version = distversion.CX_VERSION
        best_priority = -1
        for cxp in CXFIXES.cxproducts.values():
            if cxp.productid is not None and \
               re.search(cxp.productid, cx_product_id) is None:
                continue
            if cxp.version is not None and \
               re.search(cxp.version, cx_version) is None:
                continue
            if _CXID is None or best_priority < cxp.priority:
                _CXID = cxp.cxid
                best_priority = cxp.priority
                cxlog.log("found a match for cxproduct %s (%s)" % (_CXID, best_priority))
    if _CXID is None:
        cxlog.err("Could not identify the CrossOver product")
    return _CXID

def arch_add_multilib(cmd):
    if not os.path.isfile('/etc/pacman.conf'):
        return cmd

    _retcode, out, _err = cxutils.run(('grep', '-E', '\\[multilib]', '/etc/pacman.conf'), stdout=cxutils.GRAB, stderr=cxutils.NULL)
    if re.search('^# *\\[multilib]', out):
        cmd = 'sed --in-place "/^# *\\[multilib]/,/Include/ s/^# *//" /etc/pacman.conf' + '; (' + cmd + ')'
    elif re.search('^\\[multilib]', out) is None:
        cmd = '(echo "[multilib]"; echo "Include = /etc/pacman.d/mirrorlist") >>/etc/pacman.conf' + '; (' + cmd + ')'
    return cmd

def debian_add_architecture(arch, cmd):
    retcode, out, _err = cxutils.run(('dpkg', '--print-foreign-architectures'), stdout=cxutils.GRAB, stderr=cxutils.NULL)
    if retcode == 0 and out.find(arch + '\n') < 0:
        return 'dpkg --add-architecture ' + arch + ' && (' + cmd + ')'
    return cmd


#####
#
# Error management
#
#####

_ERRORS = {}

ErrorInfo = collections.namedtuple('ErrorInfo', ('level', 'title', 'lib'))

def add_error(errid, title=None, level=None, lib=None):
    _load_fixes()
    fix = CXFIXES.fixes.get(errid, None)
    if level is None:
        cxid = detect_cxproduct()
        level = CXFIXES.get_fix_level(fix, cxid)
    if title is None:
        if fix:
            title = fix.title
        if title is None:
            title = _('Unknown error %s') % errid
    if fix is None:
        pass
    elif lib is None:
        lib = fix.lib
    elif fix.lib is not None and lib != fix.lib:
        cxlog.warn("The %s fix provides %s, not the needed %s" % (errid, fix.lib, lib))
    _ERRORS[errid] = ErrorInfo(level, title, lib)

def add_masked_errors(levels=None):
    """Check if there are any other errors that may have been masked by the
    current set and which should be fixed at the same time."""
    _load_fixes()
    cxid = detect_cxproduct()
    new_errors = set(_ERRORS.keys())
    while new_errors:
        errid = new_errors.pop()
        if CXFIXES.maskedby is None or errid not in CXFIXES.maskedby:
            continue
        for maskedid in CXFIXES.maskedby[errid]:
            if maskedid in _ERRORS:
                continue
            fix = CXFIXES.fixes[maskedid]
            add = levels is None
            if not add:
                level = CXFIXES.get_fix_level(fix, cxid)
                add = level in levels
            if add:
                new_errors.add(maskedid)
                add_error(maskedid)

def remove_error(errid):
    del _ERRORS[errid]

def has_error(errid):
    return errid in _ERRORS

def get_error_level(errid):
    _load_fixes()
    fix = CXFIXES.fixes.get(errid, None)
    if fix is None:
        return None
    cxid = detect_cxproduct()
    return CXFIXES.get_fix_level(fix, cxid)

def get_error_description(errid):
    _load_fixes()
    fix = CXFIXES.fixes.get(errid, None)
    return _('Unknown error %s') % errid if fix is None else fix.description

def get_error_title(errid):
    err = _ERRORS.get(errid, None)
    return errid if err is None else err.title

def get_error_url(errid):
    # Redirect the BrokenLib issues to the MissingLib ones:
    # one page per library is more than enough pages.
    if errid.endswith('.amd64'):
        errid = errid[:-6]
    errid = errid.replace('brokenlib', 'missinglib', 1).replace('.', '-')
    return WIKI_URL + errid

def has_errors():
    if _ERRORS:
        return True
    return False

def get_error_count():
    return len(_ERRORS)

def get_errors():
    return _ERRORS

def clear_errors():
    # pylint: disable=W0603
    global _ERRORS
    _ERRORS = {}


#####
#
# Fixes management
#
#####

def strip_fix_overrides(sources, errid, key):
    bitnesses = ["32", "64"] if key[1] is None else [key[1]]
    distid = key[0]
    while distid:
        for bits in bitnesses[:]:
            ckey = (distid, bits)
            if ckey in sources and sources[ckey] != errid:
                cxlog.log_("cxfix", "   not overriding %s/%s with %s" % (ckey, sources[ckey], key))
                bitnesses.remove(bits)
                if not bitnesses:
                    return None
        distid = CXFIXES.distributions.get(distid, None).fallback

    for bits in bitnesses:
        sources[(key[0], bits)] = errid
    if len(bitnesses) == 1:
        return (key[0], bitnesses[0])
    return (key[0], None)

def get_fix(errid, distroid=None, bitness=None):
    _load_fixes()
    if errid not in CXFIXES.fixes:
        return (None, None, None)

    if CXFIXES.fixes[errid].fallback:
        # Build a unified view of the fixes for the caller.
        # Note: We rely on CXFixes.validate() to throw out infinite loops.
        merged_fixes = {}
        fix_sources = {}
        fallbackid = errid
        cxlog.log_("cxfix", "Building merged fix for %s:" % errid)
        while fallbackid:
            if fallbackid != errid:
                cxlog.log_("cxfix", " merging fixes from %s" % fallbackid)
            distfixes = CXFIXES.fixes[fallbackid].distfixes
            for key in distfixes:
                merge_key = strip_fix_overrides(fix_sources, fallbackid, key)
                if merge_key:
                    merged_fixes[merge_key] = distfixes[key]
                    cxlog.log_("cxfix", "   merged %s/%s as %s" % (key, fallbackid, merge_key))
            fallbackid = CXFIXES.fixes[fallbackid].fallback
    else:
        merged_fixes = CXFIXES.fixes[errid].distfixes

    if distroid is None:
        distroid = detect_distribution()
    if bitness is None:
        bitness = detect_bitness()

    while distroid:
        dist = CXFIXES.distributions.get(distroid, None)
        if (distroid, bitness) in merged_fixes:
            return (merged_fixes, distroid, bitness)
        if (distroid, None) in merged_fixes:
            return (merged_fixes, distroid, None)
        distroid = dist.fallback

    return (merged_fixes, None, None)

def get_distribution_property(distroid, name):
    """Returns the distribution's property, using the fallbacks if necessary.
    If there is no such property, then None is returned.
    """
    if distroid not in CXFIXES.distributions:
        return None
    dist = CXFIXES.distributions[distroid]
    while name not in dist.__dict__:
        if dist.fallback is None:
            return None
        dist = CXFIXES.distributions[dist.fallback]
    return dist.__dict__[name]

def get_packages(errid, distroid=None, bitness=None, pre=None):
    _load_fixes()

    # Return the packages for all the relevant errors
    (distfixes, fix_distroid, fix_bitness) = get_fix(errid, distroid, bitness)
    if fix_distroid is None:
        return []
    packages = []
    if pre or pre is None:
        packages = list(distfixes[(fix_distroid, fix_bitness)].prepackages)
    if not pre or pre is None:
        packages.extend(distfixes[(fix_distroid, fix_bitness)].packages)
    return packages

def get_package_key(name):
    if name.endswith(':i386'): # Debian
        return name[:-5]
    if name.endswith(':amd64'): # Debian
        return name[:-6]
    if name.endswith('.i686'): # Fedora
        return name[:-5]
    if name.endswith('.x86_64'): # Fedora
        return name[:-7]
    if name.endswith('-32bit'): # openSUSE
        return name[:-6]
    if name.startswith('lib32-'): # Arch
        return name[6:]
    return name

def get_fix_command(update=True, checklock=False, allatonce=True, yes=False):
    if not has_errors():
        return ([], None)

    distroid = detect_distribution()
    if distroid is None:
        return ([], None)
    bitness = detect_bitness()

    # First make sure the method for fixing errors in distribution is one
    # we know about. Currently that's the packagecmd method.
    pkg_cmd = get_distribution_property(distroid, 'packagecmd')
    if pkg_cmd is None:
        cxlog.warn("No command has been set for installing the packages on %s" % CXFIXES.distributions[distroid].name)
        return ([], None)
    cmd = ''
    if yes:
        if 'pacman' in pkg_cmd:
            pkg_cmd += ' --noconfirm'
        else:
            pkg_cmd += ' -y'
            if 'apt' in pkg_cmd:
                cmd = 'DEBIAN_FRONTEND=noninteractive; export DEBIAN_FRONTEND; '

    fixable_errids = []
    package_sets = {}
    has_arch32 = False
    has_deb32 = False
    prepackages = set()
    for errid in _ERRORS:
        prepackages.update(get_packages(errid, distroid, bitness, pre=True))
        fix_packages = get_packages(errid, distroid, bitness, pre=False)
        if fix_packages:
            fixable_errids.append(errid)
            package_set = set(fix_packages)

            # Group related packages into sets of packages that must be
            # installed together.
            # This is particularly important for 32- and 64-bit packages: if
            # installed separately there is a risk the 32-bit package will
            # cause the 64-bit package to be uninstalled, and then the 64-bit
            # one to cause the 32-bit package to be uninstalled. In the end
            # it will look like all commands have succeeded despite having
            # failed to get both 32- and 64-bit packages installed.
            for package in fix_packages:
                key = get_package_key(package)
                if key in package_sets:
                    package_set |= package_sets[key]
                if package.startswith('lib32-'):
                    has_arch32 = True
                if package.endswith(':i386'):
                    has_deb32 = True

            # Point all keys to the merged package set for the next merge
            for package in package_set:
                key = get_package_key(package)
                package_sets[key] = package_set

    if not package_sets:
        return ([], None)

    if prepackages:
        cmd = pkg_cmd + ' ' + cxutils.argvtocmdline(prepackages) + '; ' + cmd
    if allatonce:
        all_packages = set()
        for package_set in package_sets.values():
            all_packages.update(package_set)
        if 'libc6:i386' in all_packages:
            # Work around the "Could not perform immediate configuration on
            # 'libgcc-s1:i386'." errors. See Ubuntu bug 1871268.
            # Note that it is fine to ignore errors in this command, only the
            # next one should really succeed.
            cmd += pkg_cmd + ' -o APT::Immediate-Configure=0 libc6:i386; '
        cmd += pkg_cmd + ' ' + cxutils.argvtocmdline(sorted(all_packages))
    else:
        cmds = []
        # Avoid duplicate commands, e.g. for liblber and libldap_r
        done = set()
        for packages in package_sets.values():
            packages = sorted(packages)
            set_key = tuple(packages)
            if set_key not in done:
                cmds.append(pkg_cmd + ' ' + cxutils.argvtocmdline(packages) + ' || rc=1')
            done.add(set_key)
        cmd += 'set -x; rc=0; ' + '; '.join(cmds) + '; exit $rc'

    lockcmd = get_distribution_property(distroid, 'lockcmd')
    if checklock and lockcmd:
        cmd = lockcmd + "; " + cmd
    updatecmd = get_distribution_property(distroid, 'updatecmd')
    if update and updatecmd:
        # Ignore update errors, hoping the package list was not too outdated
        cmd = updatecmd + "; " + cmd

    if bitness == '64' and has_arch32:
        # If we need to install an Arch multilib package do make sure the
        # multilib repository is enabled.
        cmd = arch_add_multilib(cmd)
    elif bitness == '64' and has_deb32:
        # If we need to install a Debian multiarch package, make sure we have
        # the required architecture. Note that 32-bit distributions don't
        # support 64-bit packages. Note that this does not need the dpkg lock.
        cmd = debian_add_architecture('i386', cmd)

    return (fixable_errids, cmd)

def fix_errors(new_console=True, update=True, allatonce=True, gui=True, yes=True):
    # First check if there is anything that can be fixed
    fixable_errors, cmd = get_fix_command(update, checklock=True, allatonce=allatonce, yes=yes)
    if cmd is None:
        return None

    # Then determine whether to ask for confirmation before running the command
    # and adjust it if it was not yet non-interactive
    ask = (gui and yes)
    if not ask and cxproduct.is_crostini():
        # Ask for confirmation before running the command because cxsu
        # (sudo/pkexec really) is not going to give the user a chance to
        # review what's happening.
        ask = True
        fixable_errors, cmd = get_fix_command(update, checklock=True, allatonce=allatonce, yes=True)

    title = _('%(platform)s Package Installation') % {'platform': distversion.PLATFORM}
    text = _('CrossOver needs to install several %(platform)s Packages in order to run Windows applications.') % {'platform': distversion.PLATFORM}
    prompt = _('Proceed with the installation?')
    cxsu_cmd = [os.path.join(cxutils.CX_ROOT, "bin", "cxsu"),
                '--description', _('CrossOver needs your permission to install missing packages.'),
                '--console', '--ignore-home', 'sh', '-c', cmd]

    retcode = None
    if gui and ask:
        import checkgtk
        if checkgtk.check_gtk(warn_gtk=False, warn_display=False) == checkgtk.OK:
            try:
                import cxfixesdialog

                dialog = [None]
                def create_dialog(_data):
                    dialog[0] = cxfixesdialog.FixDialog(title, text, prompt, cxsu_cmd,
                                                     run_from_main_thread_callback)
                run_from_main_thread(create_dialog)
                retcode = dialog[0].retcode
                ask = False
            except:
                pass

    if ask:
        # This is also a fallback for the broken GUI case
        if refused(gui, title, text, prompt):
            return -1

    if retcode is None:
        # The packaging tools are likely to ask for confirmation on the console
        if new_console:
            stdin = subprocess.DEVNULL
        else:
            stdin = None
        ret = subprocess.run(cxsu_cmd, stdin=stdin, check=False)
        retcode = ret.returncode
        if retcode in (126, 127):
            # Detect pkexec's "dismissed dialog" exit code and treat bad
            # passwords the same way.
            retcode = -1

    if retcode == 0:
        for errid in fixable_errors:
            remove_error(errid)
    return retcode


#####
#
# Unfixed issues reporting
#
#####

def markup_error_message(message):
    if message is None:
        return None

    fixurl_re = re.compile('(?P<base>https?://[^\\s]+/)(?P<errid>\\w+) \\((?P<bitness>32|64)-bit\\)')

    def _linkify(match):
        base = match.group('base')
        urlid = errid = match.group('errid')
        if match.group('bitness') == '64':
            errid += '.amd64'
        return '<a href="%s%s">%s</a>' % (base, urlid, get_error_title(errid))

    lines = []
    for line in message.split('\n'):
        if line:
            line = fixurl_re.sub(_linkify, line)
            lines.append(line)
    return '\n'.join(lines)

def get_fix_message(distfixes, distroid, bitness, distname=True):
    cmd = get_distribution_property(detect_distribution(), 'packagecmd')
    args = [cmd]
    prepackages = distfixes[(distroid, bitness)].prepackages
    if prepackages:
        args.extend(sorted(prepackages))
        args.extend((';', cmd))
    args.extend(sorted(distfixes[(distroid, bitness)].packages))
    msg = ' '.join(args)

    if distname:
        name = CXFIXES.distributions[distroid].name
        if bitness is not None:
            name = _("%(name)s (%(bitness)s-bit)") % {'name': name, 'bitness': bitness}
        msg = "%-23s %s\n" % (name, msg)

    return msg

def get_error_message(prefix=True, verbose=False):
    if not has_errors():
        return None

    message = ""
    if prefix:
        message = _("Some errors may prevent %s from working correctly.\n") % distversion.PRODUCT_NAME


    # Get an error message for each errid
    distroid = detect_distribution()
    bitness = detect_bitness()
    errid2msg = {}
    msg2errid = {} # deduplicate issues that have identical fixes
    status2errid = {'fix': set(), 'nofix': set(), 'other': set(), 'missing': set()}
    for errid in _ERRORS:
        (distfixes, fix_distroid, fix_bitness) = get_fix(errid)
        errmsg = ''
        if distfixes is None:
            status = 'missing'
        elif fix_distroid and distfixes[(fix_distroid, fix_bitness)].packages:
            status = 'fix'
            errmsg = '  ' + get_fix_message(distfixes, fix_distroid, fix_bitness, distname=False) + '\n'
        elif fix_distroid:
            status = 'nofix'
        else:
            status = 'other'
        if verbose and distfixes:
            # Show the fixes for other platforms
            lines = []
            for (other_distroid, other_bitness) in distfixes:
                if (other_distroid != distroid or \
                    (other_bitness is not None and other_bitness != bitness)) and \
                    distfixes[(other_distroid, other_bitness)].packages:
                    lines.append(get_fix_message(distfixes, other_distroid, other_bitness))
            if lines:
                if errmsg != '':
                    errmsg += '  --- ' + _('or on other platforms:') + '\n'
                errmsg += '  ' + '  '.join(sorted(lines))

        status2errid[status].add(errid)
        errid2msg[errid] = errmsg
        if (status, errmsg) not in msg2errid:
            msg2errid[(status, errmsg)] = set()
        msg2errid[(status, errmsg)].add(errid)

    version_warning = False
    for group in ('fix', 'other', 'nofix', 'missing'):
        if not status2errid[group]:
            continue
        if group == 'fix':
            # First list the available fixes
            message += '\n' + _('The following issues can be fixed:') + '\n'
            version_warning = True
        elif group == 'other':
            # Then the issues that had no fix for this platform
            message += '\n' + _('The following issues have no known automated fix for your platform but the fixes for other platforms may help you:') + '\n'
            version_warning = True
        elif group == 'nofix':
            # Then those that are known not to be fixable on this platform
            message += '\n' + _('The following issues cannot be fixed on this platform. Check online for more information:') + '\n'
            version_warning = True
        elif group == 'missing':
            # And finally the issues that have no fix
            message += '\n' + _('The following issues need to be fixed manually. Check online for more information:') + '\n'

        added = set()
        for seed_errid in sorted(status2errid[group]):
            if seed_errid in added:
                continue

            msg = errid2msg[seed_errid]
            for errid in sorted(msg2errid[(group, msg)]):
                lib = _ERRORS[errid].lib
                if errid.endswith('.amd64'):
                    bits = '64-bit, '
                elif lib is not None or errid + '.amd64' in CXFIXES.fixes:
                    bits = '32-bit, '
                else:
                    bits = ''
                lib = '' if lib is None else '%s, ' % lib
                level = _ERRORS[errid].level
                if level is None:
                    level = _('obsolete')
                message += '* %s (%s%s%s)\n' % (get_error_url(errid), lib, bits, level)
                added.add(errid)
            if msg:
                message += msg + '\n'

    if verbose and version_warning:
        message += '\n' + _('Note that the fixes may also apply to older or newer distribution versions.') + '\n'
    return message

def report_errors(prefix=True, verbose=False, gui=True):
    if has_errors():
        message = get_error_message(prefix, verbose)
        display_error(gui, _("Could not install some %(platform)s packages") % {'platform': distversion.PLATFORM}, message)
