# (c) Copyright 2009-2013, 2015. CodeWeavers, Inc.

import os
import signal
import subprocess
import traceback

from gi.repository import Gio
from gi.repository import GdkPixbuf
from gi.repository import GLib
from gi.repository import Gtk

import cxutils
import bottlequery

import bottlecollection
import createshortcutdlg
import cxguilog
import cxguitools
import cxlog
import cxproduct
import loggingoptions
import pyop

# for localization
from cxutils import cxgettext as _


def _icon_size_list():
    return ('48x48', '256x256', '128x128', '64x64', '32x32', '16x16', '')


def _get_icon(icon_filename):
    icon = None
    if icon_filename:
        try:
            icon = GdkPixbuf.Pixbuf.new_from_file(icon_filename)
        except GLib.Error: # pylint: disable=E0712
            cxlog.warn("couldn't load icon file %s:\n%s" % (cxlog.debug_str(icon_filename), traceback.format_exc()))

    if icon is None:
        icon = cxguitools.get_std_icon('cxexe', _icon_size_list())

    if icon is not None:
        icon = icon.scale_simple(48, 48, GdkPixbuf.InterpType.BILINEAR)

    return icon


class RunCommand(Gtk.Application):

    controller = None
    bottle_name = None
    command = None

    def __init__(self):
        appid = 'com.codeweavers.' + cxproduct.get_cellar_id() + '.cxrun'
        Gtk.Application.__init__(self, application_id=appid,
                                 flags=Gio.ApplicationFlags.HANDLES_COMMAND_LINE)

        self.add_main_option(
            "bottle",
            0,
            GLib.OptionFlags.NONE,
            GLib.OptionArg.STRING,
            "Use the specified bottle. If this option is not used, fallback to $CX_BOTTLE and then to 'default'",
            "BOTTLE",
        )

        self.add_main_option(
            "command",
            0,
            GLib.OptionFlags.NONE,
            GLib.OptionArg.STRING,
            "Start the dialog with the specified command line already filled in",
            "COMMAND",
        )

        self.add_main_option(
            "allow-root",
            0,
            GLib.OptionFlags.NONE,
            GLib.OptionArg.NONE,
            "Don't warn about running this tool as root",
            None,
        )

        GLib.unix_signal_add(GLib.PRIORITY_DEFAULT, signal.SIGINT, self.quit)

    # pylint: disable=W0221
    def do_startup(self):
        Gtk.Application.do_startup(self)

    # pylint: disable=W0221
    def do_activate(self):
        # Actually show the main window
        self.controller = RunCommandController(self.bottle_name)
        if self.command:
            self.controller.set_command(self.command)

        self.add_window(self.controller.get_window())
        self.controller.present(self.bottle_name)

    def do_command_line(self, command_line):
        options = command_line.get_options_dict()
        # convert GVariantDict -> GVariant -> dict
        options = options.end().unpack()

        if 'allow-root' not in options:
            cxguitools.warn_if_root()

        if 'command' in options:
            self.command = options['command']
        else:
            arguments = command_line.get_arguments()
            if len(arguments) > 1:
                self.command = arguments[1]

        self.bottle_name = None
        if 'bottle' in options:
            self.bottle_name = options['bottle']

        if not self.bottle_name and 'CX_BOTTLE' in os.environ:
            self.bottle_name = os.environ['CX_BOTTLE']

        if not self.bottle_name and self.command:
            bottlelist = bottlequery.get_bottle_list()

            if not bottlelist:
                # This is a special case. We can't run anything until
                # a bottle is created, so launch the 'No Bottles' dialog.
                import nobottlesdlg
                nobottlesdlg.NoBottlesController(self.command)
                Gtk.main()

                return 0

            # If the launched file is inside a bottle, use that bottle
            filepath = os.path.realpath(os.path.abspath(self.command))
            for bottlename in bottlelist:
                if bottlename not in filepath:
                    continue

                cdrive = os.path.realpath(os.path.abspath(bottlequery.get_system_drive(bottlename)))
                if filepath.startswith(cdrive):
                    self.bottle_name = bottlename
                    break

            default_bottle = bottlequery.get_default_bottle()

            # Use the default bottle, if there is one:
            if not self.bottle_name and default_bottle:
                self.bottle_name = default_bottle

            # If there's only one bottle, use that.
            if len(bottlelist) == 1:
                self.bottle_name = bottlelist[0]

        if self.command and self.bottle_name:
            args = [os.path.join(cxutils.CX_ROOT, "bin", "cxstart"),
                    "--bottle", self.bottle_name, "--"]
            args.extend(cxutils.cmdlinetoargv(self.command))
            os.execv(args[0], args)
        else:
            self.activate()

        return 0


class RunCommandController:

    def __init__(self, inBottleName):
        #  Setup the GUI
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain("crossover")
        self.xml.add_from_file(cxguitools.get_ui_path("cxrun"))
        self.xml.connect_signals(self)

        #  Set up progress bar
        progbar = self.xml.get_object("ProgBar")
        progbar.set_pulse_step(.05)
        progbar.hide()

        #  Fill Bottle popup list
        bottlePopupWidget = self.xml.get_object("BottlePopup")
        self.bottleListStore = Gtk.ListStore(str, object) #display name, internal name
        bottlePopupWidget.set_model(self.bottleListStore)

        selectIter = 0
        collection = bottlecollection.sharedCollection()
        if collection.bottle_names():
            for bottleName in sorted(collection.bottle_names()):
                newrow = self.bottleListStore.append()
                self.bottleListStore.set_value(newrow, 0, bottleName)
                self.bottleListStore.set_value(newrow, 1, bottleName)
                if bottleName == inBottleName:
                    selectIter = newrow
        else:
            newrow = self.bottleListStore.append()
            self.bottleListStore.set_value(newrow, 0, _("(No bottles)"))
            self.bottleListStore.set_value(newrow, 1, None)

        if selectIter:
            bottlePopupWidget.set_active_iter(selectIter)
        else:
            bottlePopupWidget.set_active_iter(self.bottleListStore.get_iter_first())

        self.add_menu_open = False
        self.commandOp = None
        self.animateEvent = None
        self.set_widget_sensitivities(self)

        self.logging_options = loggingoptions.LoggingOptionsController(
            (inBottleName or 'log') + '.cxlog', self.xml.get_object('RunCommandDialog'))
        self.xml.get_object('DebugOptionsFrame').pack_start(self.logging_options.get_view(), True, False, 0)

        bottlePopupWidget.connect('changed', self.on_bottle_changed)

    def on_bottle_changed(self, widget):
        active_index = widget.get_active_iter()
        bottlename = self.bottleListStore.get_value(active_index, 1)
        self.logging_options.set_default_filename((bottlename or 'log') + '.cxlog')

    def get_window(self):
        return self.xml.get_object("RunCommandDialog")

    def set_command(self, command):
        self.xml.get_object("CommandEntry").set_text(command)

    def set_launcher(self, launcher):
        lnkfile = launcher.lnkfile
        if not lnkfile:
            return

        lnkfile = lnkfile.replace('/', '\\')
        lnkfile = cxutils.argvtocmdline((lnkfile,))
        self.set_command(lnkfile)

        self.xml.get_object('RunCommandBox').hide()
        self.xml.get_object('RunWithOptionsBox').show()

        self.xml.get_object('CommandName').set_label(cxutils.html_escape(launcher.menu_name()))
        self.xml.get_object('CommandBottle').set_label(launcher.parent.bottlename)
        self.xml.get_object('CommandIcon').set_from_pixbuf(
            _get_icon(launcher.get_iconfile(_icon_size_list(), 'cxexe')))

    def destroy(self):
        self.xml.get_object("RunCommandDialog").destroy()

    def cancel_clicked(self, _widget):
        self.destroy()

    def quit_requested(self, _widget):
        self.destroy()

    def progbar_pulse(self):
        progBar = self.xml.get_object("ProgBar")
        progBar.pulse()
        return True

    def set_widget_sensitivities(self, _caller):
        progBar = self.xml.get_object("ProgBar")

        collection = bottlecollection.sharedCollection()
        if self.commandOp is not None or not collection.bottle_names():
            self.xml.get_object("RunCommandButton").set_sensitive(False)
            self.xml.get_object("CommandEntry").set_sensitive(False)
            self.xml.get_object("BottlePopup").set_sensitive(False)
            self.xml.get_object("DebugOptionsFrame").set_sensitive(False)
            self.xml.get_object("CommandBrowseButton").set_sensitive(False)
            progBar.show()
        else:
            if self.xml.get_object("CommandEntry").get_text():
                self.xml.get_object("RunCommandButton").set_sensitive(True)
            else:
                self.xml.get_object("RunCommandButton").set_sensitive(False)

            self.xml.get_object("CommandEntry").set_sensitive(True)
            self.xml.get_object("BottlePopup").set_sensitive(True)
            self.xml.get_object("DebugOptionsFrame").set_sensitive(True)
            self.xml.get_object("CommandBrowseButton").set_sensitive(True)
            progBar.hide()

        if self.xml.get_object("CommandEntry").get_text():
            self.xml.get_object("CreateIconButton").set_sensitive(True)
        else:
            self.xml.get_object("CreateIconButton").set_sensitive(False)

        if not collection.bottle_names():
            self.xml.get_object("ProgBar").hide()

    def present(self, in_bottle_name):
        found = False
        iterator = self.bottleListStore.get_iter_first()
        while iterator:
            bottle_name = self.bottleListStore.get_value(iterator, 0)
            if bottle_name == in_bottle_name:
                self.xml.get_object("BottlePopup").set_active_iter(iterator)
                found = True
                break

            iterator = self.bottleListStore.iter_next(iterator)

        if not found:
            self.xml.get_object("BottlePopup").set_active_iter(
                self.bottleListStore.get_iter_first())

        self.xml.get_object("RunCommandDialog").present()

    def run_command(self, _caller):
        logFileName = ""
        loggingChannels = ""
        env_string = ""
        if self.logging_options.enable_logging:
            logFileName = self.logging_options.log_file
            self.logging_options.sanitize_channels()
            loggingChannels = self.logging_options.debug_channels
            env_string = self.logging_options.environment_variables
            self.logging_options.save_recent()

        self.animateEvent = GLib.timeout_add(100, self.progbar_pulse)

        command_line = self.xml.get_object("CommandEntry").get_text()
        command_line += ' ' + self.xml.get_object('CommandLineOptionsEntry').get_text()

        env_string += ' ' + self.xml.get_object('EnvVarEntry1').get_text()
        env_string += ' ' + self.xml.get_object('EnvVarEntry2').get_text()

        if logFileName:
            log_file = cxguilog.create_log_file(logFileName, self.get_selected_bottle(), "Running command: %s\nDebug channels: %s\nExtra environment variables: %s\n\n" % (command_line, loggingChannels, env_string))
        else:
            log_file = None

        self.commandOp = RunWineCommandOperation(self, command_line, self.get_selected_bottle().name, log_file, loggingChannels, env_string)
        self.set_widget_sensitivities(self)
        pyop.sharedOperationQueue.enqueue(self.commandOp)

    def run_command_finished(self, _runCommandOp):
        self.commandOp = None
        self.set_widget_sensitivities(self)

        GLib.source_remove(self.animateEvent)

    def get_selected_bottle(self):
        collection = bottlecollection.sharedCollection()
        bottlePopup = self.xml.get_object("BottlePopup")
        activeIndex = bottlePopup.get_active_iter()
        selectedBottleName = self.bottleListStore.get_value(activeIndex, 1)
        if selectedBottleName:
            return collection.bottle_with_name(selectedBottleName)
        return None

    def browse_for_command(self, _caller):
        filePicker = Gtk.FileChooserDialog(
            title=_("Choose a File to Run"),
            transient_for=self.xml.get_object("RunCommandDialog"),
            action=Gtk.FileChooserAction.OPEN)
        cxguitools.add_filters(filePicker, cxguitools.FILTERS_RUNNABLE | cxguitools.FILTERS_ALLFILES)

        selectedBottle = self.get_selected_bottle()
        environ = bottlequery.get_win_environ(selectedBottle.name, ("SystemDrive",))
        drive = bottlequery.expand_win_string(environ, "%SystemDrive%")
        drive = bottlequery.get_native_path(selectedBottle.name, drive)
        filePicker.set_current_folder(drive)

        filePicker.add_button(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL)
        button = filePicker.add_button(Gtk.STOCK_OPEN, Gtk.ResponseType.OK)
        button.get_style_context().add_class('suggested-action')

        if filePicker.run() == Gtk.ResponseType.OK:
            commandEntry = self.xml.get_object("CommandEntry")
            filename = cxutils.argvtocmdline((filePicker.get_filename(),))
            commandEntry.set_text(filename)

        filePicker.destroy()

    def create_icon(self, _widget):
        bottlename = self.get_selected_bottle().name
        command = self.xml.get_object("CommandEntry").get_text()
        createshortcutdlg.CreateShortcutDialog(bottlename=bottlename, command=command)

#
# Operations
#

class RunWineCommandOperation(pyop.PythonOperation):

    def __init__(self, inCommandController, inCommand, inBottleName, inLogFile=None, inDebugChannels=None, inEnv=None):
        pyop.PythonOperation.__init__(self)
        self.bottlename = inBottleName
        self.commandController = inCommandController
        self.argArray = [os.path.join(cxutils.CX_ROOT, "bin", "cxstart")]
        self.argArray.extend(["--bottle", inBottleName, '--new-console'])
        self.log_file = inLogFile
        if inDebugChannels:
            self.argArray.append("--debugmsg")
            self.argArray.append(inDebugChannels)
        if inEnv:
            self.argArray.append("--env")
            self.argArray.append(inEnv)
        cmdline_array = cxutils.cmdlinetoargv(inCommand)
        dirname = cxutils.dirname(cmdline_array[0])
        if dirname:
            self.argArray.append('--workdir')
            self.argArray.append(dirname)
        self.argArray.append("--")
        self.argArray.extend(cxutils.cmdlinetoargv(inCommand))

    def __unicode__(self):
        return "RunWineCommandOperation for " + self.bottlename


    def main(self):
        try:
            if self.log_file:
                env = os.environ.copy()
                env['CX_LOG'] = '-'
                stderr = self.log_file
            else:
                env = None
                stderr = subprocess.PIPE
            subp = subprocess.Popen(self.argArray, stdout=subprocess.PIPE, stderr=stderr, env=env, universal_newlines=True, close_fds=True) # pylint: disable=R1732
            if self.log_file:
                self.log_file.close()
            subp.communicate()
        except: # pylint: disable=W0702
            traceback.print_exc()


    def finish(self):
        self.commandController.run_command_finished(self)
        pyop.PythonOperation.finish(self)
